package gov.va.genisis2.dao.impl;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang.StringUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.orm.hibernate5.HibernateTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.dao.IUserManagementDAO;
import gov.va.genisis2.model.UserApprover;
import gov.va.genisis2.model.UserRoleType;
import gov.va.genisis2.model.Users;

/**
 * The Class UserManagementDAO.
 *
 * The UserManagement data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, UserManagement DAO
 * provide some specific data operations without exposing details of the
 * database.
 */
@Repository
@Transactional(value = "transactionManager")
public class UserManagementDAO implements IUserManagementDAO {

	/** The logger. */
	private final org.slf4j.Logger logger = LoggerFactory
			.getLogger(UserManagementDAO.class);

	/** The hibernate template. */
	@Autowired
	@Qualifier("hibernateTemplate")
	private HibernateTemplate hibernateTemplate;

	/**
	 * Gets the hibernate template.
	 *
	 * @return HibernateTemplate The HibernateTemplate.
	 */
	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}

	/**
	 * Sets the hibernate template.
	 *
	 * @param hibernateTemplate
	 *            The hibernateTemplate.
	 */
	@Autowired
	public void setHibernateTemplate(HibernateTemplate hibernateTemplate) {
		this.hibernateTemplate = hibernateTemplate;
	}

	/**
	 * This method is used to getUserDetailsByEmail.
	 * 
	 * @param email
	 *            The email.
	 * @return Users This returns Users.
	 */
	@Override
	public Users getUserDetailsByEmail(String email) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Users> query = criteriaBuilder.createQuery(Users.class);
		Root<Users> root = query.from(Users.class);

		Predicate condition = criteriaBuilder.equal(root.get("emailId"), email);
		query.where(condition);
		Users user = null;
		
		try {
			user = entityManager.createQuery(query).getSingleResult();
		} catch (NoResultException nre) {
			logger.error(CommonEnum.NO_RESULT_ENTITY.getText(), nre);
			return user;
		}

		if (user != null) {
			user.setRole(this.findUserRole(user.getUserId()));
			user.setApprover(this.findUserApprover(user.getUserId()));
		}
		return user;
	}

	/**
	 * This method is used to getUserDetailsById.
	 * 
	 * @param id
	 *            The id.
	 * @return Users This returns users.
	 */
	@Override
	public Users getUserDetailsById(int id) {
		Users user = null;
		try {
			user = hibernateTemplate.get(Users.class, id);
		} catch (NoResultException nre) {
			logger.error(CommonEnum.NO_RESULT_ENTITY.getText(), nre);
			return user;
		}

		if (user != null) {
			user.setRole(this.findUserRole(user.getUserId()));
			user.setApprover(this.findUserApprover(user.getUserId()));
		}
		return user;
	}

	/**
	 * This method is used to getUserRole.
	 * 
	 * @param uid
	 *            The uid.
	 * @return String This returns userRole.
	 */
	@Override
	public String getUserRole(int uid) {

		String userRole = StringUtils.EMPTY;

		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<UserRoleType> criteria = builder
				.createQuery(UserRoleType.class);

		Root<UserRoleType> root = criteria.from(UserRoleType.class);
		criteria.select(root);

		criteria.where(builder.and(builder.equal(root.get("users")
				.get("userId"), uid)));
		UserRoleType userRoleType = null;

		try {
			userRoleType = entityManager.createQuery(criteria)
					.getSingleResult();
		} catch (NoResultException nre) {
			logger.error(CommonEnum.NO_RESULT_ENTITY.getText() + uid, nre);
			return StringUtils.EMPTY;
		}

		if (userRoleType != null) {
			userRole = userRoleType.getRoleType().getRoleName();

		}

		return userRole;
	}

	/**
	 * This method is used to create User
	 * 
	 * @param user
	 *            The user.
	 * @return int This returns userId.
	 */
	@Override
	public int submitOrModifyUser(Users user) {
		hibernateTemplate.saveOrUpdate(user);
		return user.getUserId();
	}

	private String findUserRole(int uid) {
		String userRole = StringUtils.EMPTY;
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<UserRoleType> criteria = builder
				.createQuery(UserRoleType.class);

		Root<UserRoleType> root = criteria.from(UserRoleType.class);
		criteria.select(root);

		criteria.where(builder.and(builder.equal(
				root.get(CommonEnum.USERS.getText()).get(
						CommonEnum.USER_ID.getText()), uid)));
		UserRoleType userRoleType = null;

		try {
			userRoleType = entityManager.createQuery(criteria)
					.getSingleResult();
		} catch (NoResultException nre) {
			logger.error(CommonEnum.NO_RESULT_ENTITY.getText() + uid, nre);
			return StringUtils.EMPTY;
		}

		if (userRoleType != null) {
			userRole = userRoleType.getRoleType().getRoleName();

		}
		return userRole;
	}

	private String findUserApprover(int uid) {
		String approver = StringUtils.EMPTY;

		if (uid == 0) {
			return approver;
		}

		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<UserApprover> criteria = builder

		.createQuery(UserApprover.class);

		Root<UserApprover> root = criteria.from(UserApprover.class);
		criteria.select(root);

		criteria.where(builder.and(builder.equal(
				root.get("usersByUserId").get(CommonEnum.USER_ID.getText()),
				uid)));
		UserApprover userApprover = null;
		try {
			userApprover = entityManager.createQuery(criteria)
					.getSingleResult();
		} catch (NoResultException nre) {
			logger.error(CommonEnum.NO_RESULT_ENTITY.getText() + uid, nre);
			return StringUtils.EMPTY;
		}

		if (userApprover != null) {
			approver = userApprover.getUsersByUserApproverId().getEmailId();

		}
		return approver;
	}

}
